#include <stdio.h>          /* printf */
#include <stdlib.h>         /* exit */
#include <unistd.h>         /* read */
#include <string.h>         /* strtok, sprintf */
#include <netdb.h>
#include <pthread.h>        /* pthread_create */
#include <sys/socket.h>     /* socket, setsockopt */
#include <math.h>           /* pow, sqrt */
#include <ctype.h>          /* tolower */

/* Estructuras */
typedef enum TipoFiltroE {      \
    BUTTERWORTH,                \
        CHEBYSCHEV,             \
            BESSEL,             \
                UNKNOWN         \
} TipoFiltro;


typedef struct FiltroP {
    TipoFiltro      tipo;
    int             frecuencia_corte;
    int             orden;
} filtro_t;


/* Defines */
#define MAX_RECV_BUFFER     (200)
#define USER_SIZE           (200)
#define NETLIST_SIZE        (1000)
#define PI                  (3.14159)


/* Prototipos de Funciones */
void * servicio(void *sockcl_void);


int main(int argc, char * argv[]) 
{
    int                     sockfd;
    int                     sockcl;
    struct sockaddr_in      address;
    pthread_t               thread;
    int                     domain          = AF_INET;
    int                     type            = SOCK_STREAM;
    int                     opt             = 1;
    int                     addrlen         = sizeof(address);
    int                     port            = 1501;
    int                     buffer_size     = 4;
    
    /* Creación del descriptor del Socket */
    if( (sockfd = socket(domain, type, 0)) == 0 ) 
    {
        printf("Problemas al crear el socket.\n");
        exit(1);
    }
    
    
    /* Configuración de la opciones para el Socket */
    if( setsockopt(sockfd, SOL_SOCKET, SO_REUSEADDR, &opt, sizeof(opt)) != 0 ) 
    {
        printf("Problemas al condifurar el socket.\n");
        exit(1);
    }

    
    /* Configuración de la estructura address */
    address.sin_family         = domain;
    address.sin_addr.s_addr    = INADDR_ANY;
    address.sin_port           = htons(port);

    /* Enlazando el socket */
    if ( bind(sockfd, (struct sockaddr*)&address, addrlen) < 0 ) 
    {
        printf("Problemas al enlazar el socket.\n");
        exit(1);
    }
    
    /* Bucle Principal del Servidor */
    while( listen(sockfd, buffer_size) >= 0 ) 
    {
        if( (sockcl = accept(sockfd, (struct sockaddr*)&address, (socklen_t*)&addrlen)) < 0 ) 
        {
            printf("Problemas al aceptar el socket.\n");
            exit(1);
        }
        if( pthread_create(&thread, NULL, servicio, (void *)&sockcl) != 0 ) 
        {
            printf("Problemas al crear el hijo.\n");
            exit(1);
        }
    }

    return 0;
}

/******************************************************************************************************************
* Tareas Algoritmo Genético: Esta función procesa el paquete recibido y genera una base de datos con las
*                            netlist enviadas por el algoritmo genético. Además genera las gestiones necesarias
*                            para que interfaz_ngspice realice la simulación de los circuitos.
*
* @param packet paquete de datos recibido por el socket
*               
* La estructura de los paquetes es la siguiente:
*
*   [Comando] [Usuario] [Generacion] [Id] [Filtro] [Fenotipo]
*
*   Los parámetros llegan en formato ASCII separados por 
*   saltos de línea '\n'
*/
void TareasAG(char* packet, int sockcl)
{    
    char        Usuario[USER_SIZE];         ///< Nombre del Usuario de la Simulación
    char        netlist[NETLIST_SIZE];      ///< Netlist para realizar la simulación              
    int         Generacion;                 ///< Generación a Simular
    int         Id;                         ///< Id del Circuito a Simular
    filtro_t    Filtro;                     ///< Filtro con el que comparar la simulación
    FILE *      destFile;                   ///< Archivo .cir para realizar la simulación
    FILE *      simulationResultFile;       ///< Archivo .data con resultados de la simulación
    char        filepath[200];              ///< Ruta del archivo .cir
    char        simulationPath[200];        ///< Ruta del archivo .data        
    char        command[200];               ///< Comando para ejecutar la interfaz ngspice
    char *      token;                      ///< Puntero para parsear la data en packet
    float       ECM;                        ///< Error Cuadrático Medio
    int         cont;                       ///< Numero de muestras comparadas
    char        msgToSend[30];              ///< Mensaje a enviar al AG

    // Puntero al comando
    token = strtok(packet, "\n");
    
    // Puntero al usuario
    token = strtok(NULL, "\n");
    sprintf(Usuario, "%s", token);
    for(int i=0; i<strlen(Usuario); i++)
    {
        // Pasamos todo a LowerCase porque ngspice usa así los archivos .data
        Usuario[i] = tolower(Usuario[i]);
    }

    // Puntero a la generación
    token = strtok(NULL, "\n");
    Generacion = atoi(token);

    // Puntero al id
    token = strtok(NULL, "\n");
    Id = atoi(token);

    // Puntero al tipo del filtro
    token = strtok(NULL, "\n");
    char* filter_token = strtok(token, " ");
    
    if( strcmp(filter_token, "butter") == 0 ) {
        Filtro.tipo = BUTTERWORTH;
        
        // Puntero a la frecuencia de corte del filtro
        filter_token = strtok(NULL, " ");
        Filtro.frecuencia_corte = atoi(filter_token);

        // Puntero al orden del filtro
        filter_token = strtok(NULL, " ");
        Filtro.orden = atoi(filter_token);
    }
    else if( strcmp(filter_token, "chevy") == 0 )   Filtro.tipo = CHEBYSCHEV;
    else if( strcmp(filter_token, "bessel") == 0 )  Filtro.tipo = BESSEL;
    else                                            Filtro.tipo = UNKNOWN;
  
    // TODO: Implementar variables importantes para los demás filtros

    // Puntero a la netlist
    token = filter_token + strlen(filter_token) + 1;
    sprintf(netlist, "%s", token);

    printf("TAREA ALGORITMO GENETICO:\n\n");
    printf("Usuario: %s\n", Usuario);
    printf("Generacion: %d\n", Generacion);
    printf("Id: %d\n", Id);
    printf("Filtro: %d, %d Hz, %d\n", (int) Filtro.tipo, Filtro.frecuencia_corte, Filtro.orden);
    printf("Netlist: \n%s\n\n", token); fflush(stdout);
    
    // Creamos el archivo .cir para que el ngspice pueda hacer la simulación
    sprintf(filepath, "%s-%d-%d.cir", Usuario, Generacion, Id);
    destFile = fopen(filepath, "wb");

    fprintf(destFile, "%s Circuit Generation %d Id %d\n\n", Usuario, Generacion, Id); 
    fprintf(destFile, "%s\n\n", netlist);
    fprintf(destFile, ".control\n\n");
    fprintf(destFile, "ac dec 10 1 10k\n");
    fprintf(destFile, "gnuplot %s-%d-%d V(2)/V(1) xlimit 1 10k \n\n", Usuario, Generacion, Id);
    fprintf(destFile, ".endc\n");
    fprintf(destFile, ".end\n");

    fclose(destFile);
    
    // Ejecutamos la interfaz ngspice
    sprintf(command, "./interfaz %s", filepath);
    system(command);

    // Cálculo del error cuadrático medio
    ECM     = 0.0;
    cont    = 0;
    sprintf(simulationPath, "%s-%d-%d.data", Usuario, Generacion, Id);
    simulationResultFile = fopen(simulationPath, "r");

    if( simulationResultFile == NULL)
    {
        printf("Error al abrir: %s\n", simulationPath);
        ECM = 10.0;
    }
    else
    {
        char line[200];
        while( fgets(line, 200, simulationResultFile) != NULL )
        {
            float   frecuencia;
            float   magnitud_real;
            float   magnitud_teorica;
            char *  pend;
            
            // Obtenemos los valores reales de la simulación
            frecuencia = strtof(line, &pend);
            magnitud_real = strtof(pend, NULL);

            // Obtenemos el valor teórico de la magnitud (solo butterworth) TODO: Hacer que funcione con los otros tipos de filtro.
            magnitud_teorica = 1.0/sqrt(1+pow(frecuencia/Filtro.frecuencia_corte, 2*Filtro.orden));

            // Acumular Error cuadrático en ECM
            float error = magnitud_teorica - magnitud_real;
            ECM += sqrt(pow(error, 2)); // pow(magnitud_teorica - magnitud_real, 2);
            cont++;
        }
        fclose(simulationResultFile);
        // Obtenemos el Error Cuadrático Medio
        ECM /= cont;
    }

    sprintf(msgToSend, "%.4f", ECM);
    send(sockcl, msgToSend, 30, 0);

    printf("ECM: %.4f\n", ECM);
    
}

/******************************************************************************************************************
* Callback principal de los hilos del servidor.
*
* @param sockcl_void es el descriptor del socket del cliente
*/
void * servicio(void *sockcl_void) {
    char    msgRecv[MAX_RECV_BUFFER];
    char    comando;
    int *   sockcl = (int *) sockcl_void;

    read(*sockcl, msgRecv, MAX_RECV_BUFFER);
    comando = msgRecv[0];
    
    switch( comando ) {
        case 'G': {
            /* La trama proviene del AG */
            TareasAG(msgRecv, *sockcl);
        }
        break;

        case 'I': {
            /* La trama proviene de la interfaz principal */
        }
        break;

        default: {
            /* Se muestra un error en pantalla */
            printf("ERROR: Comando inválido en la trama.\n");
        }
        break;
    }
    printf("Mensaje recibido: %s.\n", msgRecv);
    fflush(stdout);
}
