/* This class encapsulates common gates features and behavoirs.
   Any gate has one output connected to any number of inputs (drains).
   Any gate has its delay.
   Each gate has a reference to a Simulator objeto to
   register future events. 
   The logic of each gate must be defined in a
   specific inherited class */ 
import java.awt.Graphics2D;
import java.awt.geom.Path2D;
import java.awt.geom.Rectangle2D;
import java.util.*;

public abstract class Gate implements UpdateListener, InSignal, OutSignal{
  private final int id;  // id for a gate within its class
  private float delay, Delay;
  private boolean alreadyScheduledValue, outputValue;
  private ArrayList <InSignal> drains;
  private Simulator simulator, sim;
  
  protected Gate (float d, int nextId, Simulator s){
    sim = s;
    Delay = d;
    id = nextId;
    drains = new ArrayList<InSignal>();
    resetGate();
  }
  public abstract boolean computeOutput();
    
  // UpdateListener interface
  public void updateOutputValue(boolean newValue){
     outputValue = newValue;
  }
  public void propagateOutputValue(){
     for (InSignal drain: drains) 
       drain.inputValueChanged();
  }
  // end UpdateListener

  // InSignal interface
  public void inputValueChanged() {
     boolean newValue= computeOutput();
     if (newValue != alreadyScheduledValue) {
       ChangeEvent e = new ChangeEvent(this, newValue, simulator.getTime()+delay);
       simulator.add(e);
       alreadyScheduledValue = newValue;
     }
  }
  // end InSignal
  
  // OutSignal interface
  public boolean getValue() {
     return outputValue;
  }  
  public void connectOutput(InSignal drain) {
    if (drain != null) {
      drains.add(drain); 
      drain.addSource(this);
    }
  }
  public void connectDrain(InSignal drain) {
    if (drain!=null) drains.add(drain);
  }
  public void removeFromDrains(InSignal drain) {
    if (drain!=null) drains.remove(drain);
  }
  public void disconnectOutput(InSignal drain) {
     if (drain != null){
       drains.remove(drain);
       drain.removeFromSource(this);
     }
  }
  public void disconnectAllOutput() {
    while(drains.size()!=0) {
      drains.get(0).removeFromSource(this);
      drains.remove(0);
    }
  }
  public String getDescription () {
     return getClass().getSimpleName().substring(1)+"_"+id;
  }
  // end OutSignal
  public void resetGate(){
	  	simulator = sim;
	  	delay = Delay;	    
	    alreadyScheduledValue = outputValue = true;
	    ChangeEvent e = new ChangeEvent(this, alreadyScheduledValue, simulator.getTime()+delay);
	    simulator.add(e);
  }
  
    
} 